package custom.checkers;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import chemaxon.checkers.CheckerInfo;
import chemaxon.checkers.ExternalStructureChecker;
import chemaxon.checkers.Persistent;
import chemaxon.checkers.result.DefaultExternalStructureCheckerResult;
import chemaxon.checkers.result.StructureCheckerResult;
import chemaxon.struc.MolAtom;
import chemaxon.struc.MolBond;
import chemaxon.struc.Molecule;
import chemaxon.struc.RxnMolecule;

@CheckerInfo(
		name = "Duplicate Atom Map Checker",
		description = "Checks for mapping duplicates in a reaction.",
		noErrorMessage = "No duplicate mappings found",
		moreErrorMessage = "duplicate mappings found",
		editorClassName = "custom.checkers.DuplicateAtomMapCheckerEditor",
		actionStringToken="duplicateatommap")
public class DuplicateAtomMapChecker extends ExternalStructureChecker {

	/** error code of duplicate atom map checker */
	public static final String DUPLICATE_ATOM_MAP_CHECKER_ERROR = "duplicateAtomMapCheckerError";

	/** property key for reaction side */
	public static final String REACTION_SIDE = "reactionSide";

	/**
	 * Specifies the reaction side to check for duplicate mappings.
	 */
	public enum ReactionSide {

		/** check the reactant side only */
		REACTANT,

		/** check the product side only */
		PRODUCT,

		/** check both sides */
		BOTH
	}

	@Persistent
	private ReactionSide reactionSide = ReactionSide.BOTH;
	
	public DuplicateAtomMapChecker() {
		super(DUPLICATE_ATOM_MAP_CHECKER_ERROR);
	}
	
	/**
	 * Constructs a duplicate atom map checker with specified settings.
	 * 
	 * @param params
	 *            the settings to use
	 */
	// NOTE: this constructor is required by StructureCheckerFactory if checker
	// has parameters.
	public DuplicateAtomMapChecker(Map<String, String> params) {
		super(DUPLICATE_ATOM_MAP_CHECKER_ERROR);
		this.reactionSide = ReactionSide.BOTH;
		if (params.containsKey(REACTION_SIDE)) {
			String value = params.get(REACTION_SIDE).toUpperCase();
			try {
				this.reactionSide = ReactionSide.valueOf(value);
			} catch (IllegalArgumentException e) {
				// invalid argument set, using default
			}
		}
	}
	
	/**
	 * Returns the reaction side to check for duplicate mappings.
	 * 
	 * @return the reaction side to check for duplicate mappings
	 */
	public ReactionSide getReactionSide() {
		return reactionSide;
	}

	/**
	 * Sets the reaction side to check for duplicate mappings.
	 * 
	 * @param reactionSide
	 *            the reaction side to check
	 */
	public void setReactionSide(ReactionSide reactionSide) {
		ReactionSide oldValue = getReactionSide();
		this.reactionSide = reactionSide;
		propertyChangeSupport.firePropertyChange(REACTION_SIDE, oldValue,
				reactionSide);

	}
	
	@Override
	protected StructureCheckerResult check1(Molecule molecule) {
		// we are checking only reactions
		if (molecule.isReaction()) {

			// create a list for atoms
			List<MolAtom> atomList = new ArrayList<MolAtom>();

			if (ReactionSide.REACTANT.equals(getReactionSide())
					|| ReactionSide.BOTH.equals(getReactionSide())) {
				// if we are checking reactants, add the duplicates to the list
				atomList.addAll(getAtomsWithMappingDuplicates(RxnMolecule
						.getReaction(molecule).getReactants()));
			}
			if (ReactionSide.PRODUCT.equals(getReactionSide())
					|| ReactionSide.BOTH.equals(getReactionSide())) {
				// if we are checking products, add the duplicates to the list
				atomList.addAll(getAtomsWithMappingDuplicates(RxnMolecule
						.getReaction(molecule).getProducts()));
			}
			if (!atomList.isEmpty()) {
				// create and return the result
				return new DefaultExternalStructureCheckerResult(this,
						atomList, Collections.<MolBond> emptyList(), molecule,
						DUPLICATE_ATOM_MAP_CHECKER_ERROR);
			}
		}

		// return with no result
		return null;
	}

	/**
	 * Returns a list of atoms that have the same mapping in the input set.
	 * 
	 * @param molecules
	 *            the input set
	 * @return a list of atoms that have the same mapping in the input set
	 */
	protected static List<MolAtom> getAtomsWithMappingDuplicates(
			Molecule[] molecules) {

		// create a list for results
		List<MolAtom> list = new ArrayList<MolAtom>();

		// create a map for mapping - atom data
		Map<Integer, MolAtom> mappings = new HashMap<Integer, MolAtom>();

		// for each molecule in the input set
		for (Molecule molecule : molecules) {

			// iterate all atoms in the molecule
			for (MolAtom atom : molecule.getAtomArray()) {
				int atomMap = atom.getAtomMap(); // get the atom map

				// if atom has mapping
				if (atomMap != 0) {

					// check if mapping already found
					if (mappings.containsKey(atomMap)) {
						// if the list not contains the other atom with same
						// mapping, add it to the list
						if (!list.contains(mappings.get(atomMap))) {
							list.add(mappings.get(atomMap));
						}
						list.add(atom); // add atom to the error list
					} else {
						mappings.put(atomMap, atom); // add mapping to the
						// mappings set
					}
				}
			}
		}

		// return the result
		return list;
	}
	
}
