package chemaxon.standardizer.external;

import java.util.Map;

import chemaxon.calculations.stereo.Stereochemistry;
import chemaxon.standardizer.AbstractStandardizerAction;
import chemaxon.standardizer.Changes;
import chemaxon.standardizer.StandardizerActionInfo;
import chemaxon.struc.Molecule;
import chemaxon.struc.StereoConstants;

/**
 * Set chiral flag standardizer action
 * 
 * @author IT
 */
@StandardizerActionInfo(
	name = "Set Chiral Flag", 
	description = "Sets the chiral flag if the molecule contains exactly one stereo center", 
	actionStringToken = "setchiralflag", 
	xmlToken = "SetChiralFlag", 
	iconPath = "/chemaxon/standardizer/external/setChiralFlag24.png",
	helpText = "Sets the chiral flag if the molecule contains exactly one stereo center")
public class SetChiralFlagAction extends AbstractStandardizerAction {

    /**
     * Initializes a set chiral flag action
     * @param params the parameters of the action
     */
    public SetChiralFlagAction(Map<String, String> params) {
	super(params);
	// no additional parameters needs to be evaluated
    }

    @Override
    protected Changes standardize1(Molecule molecule) throws IllegalArgumentException {
	// Collect chiral centers
	Stereochemistry stereo = new Stereochemistry();
	stereo.setMolecule(molecule);
	int[] stereoCenters = stereo.chiralCenters();
	// If stereo center count is not 1, the molecule should not be marked as
	// absolute stereo
	if (stereoCenters == null || stereoCenters.length != 1) {
	    molecule.setAbsStereo(false);
	    return null;
	}
	// Check whether the center is marked correctly
	int chirality = molecule.getChirality(stereoCenters[0]);
	molecule.setAbsStereo(chirality == StereoConstants.CHIRALITY_R || chirality == StereoConstants.CHIRALITY_S);
	// atoms were not modified, no changes need to be collected
	return null;
    }
}
