package chemaxon.standardizer.external;

import java.util.Map;

import chemaxon.standardizer.AbstractStandardizerAction;
import chemaxon.standardizer.Changes;
import chemaxon.standardizer.Persistent;
import chemaxon.standardizer.StandardizerActionInfo;
import chemaxon.standardizer.actions.ReplaceAtomsAction;
import chemaxon.struc.MolAtom;
import chemaxon.struc.Molecule;

/**
 * Test and demo external standardizer action
 * 
 * @author Imre Barna
 * @since 6.0
 */
@StandardizerActionInfo(
	name = "Remove Charge", 
	description = "Removes charges from the atoms of the molecule", 
	actionStringToken = "removecharge", 
	xmlToken = "RemoveCharge", 
	editorClassName = "com.chemaxon.standardizer.external.RemoveChargeActionEditor",
	iconPath = "/com/chemaxon/standardizer/external/icon.png", 
	helpText = "<html>Removes charges from the atoms of the molecule<br />"
	+ "Options:<ul>"
	+ "<li>valence error checked: removes charges only in case of the resulted molecule is of valid valence</li>"
	+ "<li>valence error not checked: removes all charges anyway</li>" + "</ul></html>")
public class RemoveChargeAction extends AbstractStandardizerAction {

    /** Property key for valence. Value is of {@link Boolean} */
    public static final String PROPERTY_KEY_CHECK_VALENCE = "checkvalence";
    @Persistent(alias = PROPERTY_KEY_CHECK_VALENCE)
    private boolean checkValence = false;

    /**
     * Initializes a remove charge action
     * 
     * @param params
     *            the parameters of the action
     */
    public RemoveChargeAction(Map<String, String> params) {
	super(params);
	if (params.containsKey(PROPERTY_KEY_CHECK_VALENCE)) {
	    checkValence = Boolean.parseBoolean(params.get(PROPERTY_KEY_CHECK_VALENCE));
	}
    }

    /**
     * Sets whether the action should check for valence error when removing
     * charges, and only apply changes in case of valid valence
     * 
     * @param checkValence
     *            whether the action should check for valence error when
     *            removing charges
     */
    public void setCheckValence(boolean checkValence) {
	boolean oldValue = this.checkValence;
	this.checkValence = checkValence;
	support.firePropertyChange(PROPERTY_KEY_CHECK_VALENCE, oldValue, checkValence);
    }

    /**
     * Gets whether the action checks for valence error when removing charges,
     * and only apply changes in case of valid valence
     * 
     * @return whether the action checks for valence error when removing charges
     */
    public boolean isCheckValence() {
	return checkValence;
    }

    @Override
    protected Changes standardize1(Molecule molecule) throws IllegalArgumentException {
	Changes changes = new Changes(molecule);
	MolAtom[] atoms = molecule.getAtomArray();
	for (int i = 0; i < atoms.length; i++) {
	    MolAtom atom = atoms[i];
	    if (atom.getCharge() != 0) {
		if (checkValence) {
		    Molecule clone = molecule.cloneMolecule();
		    clone.getAtom(i).setCharge(0);
		    clone.valenceCheck();
		    if (clone.hasValenceError()) {
			continue;
		    }
		}
		atom.setCharge(0);
		changes.modifyAtom(atom);
	    }
	}
	return changes;
    }
    
    @Override
    public RemoveChargeAction clone() throws CloneNotSupportedException {
	final RemoveChargeAction clone = (RemoveChargeAction) super.clone();
	clone.checkValence = checkValence;
	return clone;
    }

}
