module "generic_container_def" {
  source  = "cloudposse/ecs-container-definition/aws"
  version = "0.58.1"

  container_name  = local.container_name
  container_image = local.container_image
  # Need to set container cpu, otherwise there is a noticeable performance loss.
  container_cpu                = var.task_cpu
  container_memory             = var.task_memory - 4
  container_memory_reservation = var.task_memory_reservation

  healthcheck = {
    command     = ["CMD-SHELL", "curl -f http://localhost:${var.webservice_port}${var.healthcheck_uri} || exit 1"]
    interval    = 30
    retries     = var.retries
    startPeriod = var.grace_period
    timeout     = 5
  }

  port_mappings = [
    {
      containerPort = var.webservice_port
      hostPort      = var.webservice_port
      protocol      = "tcp"
    }
  ]

  environment = module.environment_helper.environment
  secrets     = module.environment_helper.secrets

  log_configuration = {
    logDriver = "awslogs"
    options = {
      awslogs-group         = var.loggroup_name
      awslogs-region        = data.aws_region.current.name
      awslogs-stream-prefix = local.container_name
    }
    secretOptions = null
  }

  start_timeout = 30
  stop_timeout  = 30
}

module "environment_helper" {
  source = "./environment-helper"

  environment = merge({
    "CHEMAXON_LICENSE_URL" : data.aws_ssm_parameter.chemaxon_license_url.value,
    "JDBC_URL" : local.rds_db_url,
    "JDBC_USER" : data.aws_ssm_parameter.db_user.value,
    "JDBC_PASSWORD" : data.aws_ssm_parameter.db_password.value,
    "LABEL_CACHE_POLICY" : "LRU",
    "LABEL_CACHED_OBJECT_COUNT" : "8000000",
    "MOLECULE_CACHE_POLICY" : "LRU",
    "MOLECULE_CACHED_OBJECT_COUNT" : "8000000",
    "FINGERPRINT_CACHED_OBJECT_COUNT" : "8320000",
    "XMX" : "11839m",
    "LOG_LEVEL" : "INFO",
    "LOG" : true,
  }, var.task_environment_variables)
}

resource "aws_ecs_task_definition" "this" {
  count = var.desired_count > 0 ? 1 : 0

  family                = "${var.webservice_fullname}-${var.env_name}"
  container_definitions = module.generic_container_def.json_map_encoded_list
  execution_role_arn    = aws_iam_role.taskexec_role.arn
  task_role_arn         = var.task_role_arn

  cpu                      = var.task_cpu
  memory                   = var.task_memory
  network_mode             = "awsvpc"
  requires_compatibilities = ["FARGATE"]

  tags = merge(var.tags, { "Name" = "${var.webservice_fullname}-${var.env_name}" })
}

resource "aws_ecs_service" "this" {
  count = var.desired_count > 0 ? 1 : 0

  name                               = local.container_name
  cluster                            = aws_ecs_cluster.cluster.id
  task_definition                    = aws_ecs_task_definition.this.0.arn
  desired_count                      = var.desired_count
  deployment_maximum_percent         = var.deployment_maximum_percent
  deployment_minimum_healthy_percent = var.deployment_minimum_healthy_percent
  enable_ecs_managed_tags            = true
  health_check_grace_period_seconds  = var.grace_period
  propagate_tags                     = "SERVICE"

  capacity_provider_strategy {
    capacity_provider = var.capacity_provider
    weight            = 1
  }

  deployment_controller {
    type = var.deployment_controller
  }

  dynamic "load_balancer" {
    for_each = aws_lb_target_group.blue
    content {
      target_group_arn = load_balancer.value.arn
      container_name   = local.container_name
      container_port   = var.webservice_port
    }
  }

  network_configuration {
    subnets         = data.aws_subnets.private_subnets.ids
    security_groups = [aws_security_group.choral_service_sg.id, aws_security_group.rds_security_group.id]
  }

  dynamic "service_registries" {
    for_each = var.service_discovery_namespace_id
    content {
      registry_arn   = aws_service_discovery_service.this[0].arn
      container_name = local.container_name
    }
  }

  tags = merge(var.tags, { "Name" = local.container_name })
}

resource "aws_lb_target_group" "blue" {
  count = var.desired_count > 0 ? 1 : 0

  name        = "${local.container_name}-${var.env_name}-blue"
  port        = var.webservice_port
  protocol    = "HTTP"
  target_type = "ip"
  vpc_id      = data.aws_vpc.vpc.id

  health_check {
    path                = var.healthcheck_uri
    interval            = var.health_check_interval
    unhealthy_threshold = var.unhealthy_threshold
  }

  tags = merge(var.tags, { "Name" = "${local.container_name}-${var.env_name}-blue" })
}

resource "aws_service_discovery_service" "this" {
  count = var.desired_count > 0 && length(var.service_discovery_namespace_id) != 0 ? 1 : 0

  name = local.container_name

  dns_config {
    namespace_id = var.service_discovery_namespace_id[0]

    dns_records {
      ttl  = 10
      type = "A"
    }

    routing_policy = "MULTIVALUE"
  }

  health_check_custom_config {
    failure_threshold = 1
  }
}
