/*
 * Copyright (c) 1998-2011 ChemAxon Ltd. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * ChemAxon. You shall not disclose such Confidential Information
 * and shall use it only in accordance with the terms of the agreements
 * you entered into with ChemAxon.
 *
 */
package example.services;

import chemaxon.formats.MolFormatException;
import chemaxon.formats.MolImporter;
import chemaxon.marvin.services.localservice.Alias;
import chemaxon.marvin.services.localservice.Description;
import chemaxon.struc.Molecule;


/**
 * This is a sample class to demonstrate how to write
 * classes for Marvin Services Local Service implementation.
 * @author Istvan Rabel
 */

public class SampleService {

    /**
     * Returns the number of atoms in the specified molecule
     * @param molecule the molecule being checked
     * @return the number of atoms in the molecule
     */
    /* 
     * (non-javadoc)
     * This method can be called as a LocalService from
     * Marvin Sketch, cxcalc and Chemical Terms.
     * Annotations are used to provide default names
     * for Service and arguments, as well as a description.
     */
    @Alias(name="AtomCount", params={"Structure"})
    @Description("Returns the number of atoms in the structure")
    public Integer countAtoms(Molecule molecule) {
	return molecule.getAtomCount();
    }
    
    /**
     * Returns a formatted (HTML) message with the number of
     * atoms in the molecule imported from argument.
     * @param moleculeString a string representation of a molecule
     * @return a formatted (HTML) message with the number of atoms
     */
    /* 
     * (non-javadoc)
     * This method can be called as a LocalService from
     * Marvin Sketch, cxcalc and Chemical Terms.
     * Annotations are used to provide default names
     * for Service and arguments, as well as a description.
     */
    @Alias(name="AtomCountText", params={"Molecule"})
    @Description("Returns a formatted text message containing the number of atoms in the structure.")
    public String countAtomsHTML(String moleculeString) {
	// import the molecule
	Molecule molecule = null;
	try {
	    molecule = MolImporter.importMol(moleculeString);
	} catch (MolFormatException e) {
	    // invalid molecule string
	    molecule = new Molecule();
	}
	
	// get the atom count
	int value = countAtoms(molecule);
	
	// build and return the result string
	StringBuilder builder = new StringBuilder("<html><body>");
	if(value > 1) {
	    builder.append("The structure has <font color='blue'><b>");
	    builder.append(value);
	    builder.append("</b></font> atoms.");
	} else {
	    builder.append("The structure has <font color='red'><i>"
		    + (value == 0 ? "no atoms" : "only one atom")
		    + "</i></font>.");
	}
	builder.append("</body></html>");
	return builder.toString();
    }
    
}
